<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class InventoryController extends Controller
{
    public function categories()
    {
        return response()->json(
            ProductCategory::orderBy('name')->get()
        );
    }

    public function summary()
    {
        $totalProducts = Product::count();
        $totalItems = ProductItem::where('status', 'in_stock')->count();
        $lowStock = Product::whereColumn('stock_qty', '<=', 'low_stock_alert')->count();

        return response()->json([
            'total_products' => $totalProducts,
            'total_items' => $totalItems,
            'low_stock' => $lowStock,
        ]);
    }

    public function listProducts()
    {
        $products = Product::with('category')
            ->withCount([
                'items as serial_in_stock_count' => fn($q) => $q->where('status', 'in_stock'),
            ])
            ->orderBy('name')
            ->get();

        return response()->json($products);
    }

    public function lookupSerial(string $serial)
    {
        $serial = strtoupper(trim($serial));
        $item = ProductItem::where('serial_number', $serial)->first();
        if (!$item) {
            return response()->json(['message' => 'serial_not_found'], 404);
        }
        if ($item->status !== 'in_stock') {
            return response()->json(['message' => 'serial_not_in_stock'], 422);
        }

        $product = Product::find($item->product_id);
        if (!$product) {
            return response()->json(['message' => 'product_not_found'], 404);
        }

        return response()->json([
            'serial' => $item->serial_number,
            'product' => $product,
        ]);
    }

    public function storeCategory(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:190', 'unique:product_categories,name'],
        ]);

        $cat = ProductCategory::create([
            'name' => $data['name'],
        ]);

        return response()->json($cat, 201);
    }

    public function storeProduct(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:190'],
            'category_id' => ['required', 'integer', 'exists:product_categories,id'],
            'model' => ['nullable', 'string', 'max:190'],
            'size' => ['nullable', 'string', 'max:190'],
            'details' => ['nullable', 'string'],
            'sku' => ['nullable', 'string', 'max:190'],
            'barcode' => ['nullable', 'string', 'max:190'],
            'cost_price' => ['nullable', 'numeric', 'min:0'],
            'selling_price' => ['nullable', 'numeric', 'min:0'],
        ]);

        $sku = $data['sku'] ?? $this->generateSku($data['name']);

        $product = Product::create([
            'name' => $data['name'],
            'sku' => $sku,
            'barcode' => $data['barcode'] ?? null,
            'cost_price' => $data['cost_price'] ?? 0,
            'selling_price' => $data['selling_price'] ?? 0,
            'stock_qty' => 0,
            'low_stock_alert' => 5,
            'is_active' => true,
            'category_id' => $data['category_id'],
            'model' => $data['model'] ?? null,
            'size' => $data['size'] ?? null,
            'details' => $data['details'] ?? null,
        ]);

        return response()->json($product, 201);
    }

    public function searchProducts(Request $request)
    {
        $categoryId = $request->query('category_id');
        $name = $request->query('name');
        $model = $request->query('model');

        $q = Product::query();
        $q->when($categoryId, fn($qq) => $qq->where('category_id', $categoryId));
        $q->when($name, fn($qq) => $qq->where('name', 'like', '%'.$name.'%'));
        $q->when($model, fn($qq) => $qq->where('model', 'like', '%'.$model.'%'));

        return response()->json($q->orderBy('name')->limit(200)->get());
    }

    public function addItems(Request $request, Product $product)
    {
        $data = $request->validate([
            'serial_numbers' => ['required', 'array', 'min:1'],
            'serial_numbers.*' => ['required', 'string', 'max:190'],
        ]);

        $raw = $data['serial_numbers'];
        $normalized = array_values(array_unique(array_map(fn($s) => strtoupper(trim($s)), $raw)));
        $normalized = array_values(array_filter($normalized, fn($s) => $s !== ''));

        if (count($normalized) === 0) {
            return response()->json([
                'added_count' => 0,
                'skipped_serials' => [],
                'total_received' => 0,
            ]);
        }

        $existing = ProductItem::whereIn('serial_number', $normalized)
            ->pluck('serial_number')
            ->all();

        $existingMap = array_fill_keys($existing, true);
        $toInsert = [];
        $skipped = [];

        foreach ($normalized as $serial) {
            if (isset($existingMap[$serial])) {
                $skipped[] = $serial;
                continue;
            }
            $toInsert[] = [
                'product_id' => $product->id,
                'serial_number' => $serial,
                'status' => 'in_stock',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        $addedCount = 0;
        if (!empty($toInsert)) {
            DB::transaction(function () use ($product, $toInsert, &$addedCount) {
                ProductItem::insert($toInsert);
                $addedCount = count($toInsert);
                $product->increment('stock_qty', $addedCount);
            });
        }

        return response()->json([
            'added_count' => $addedCount,
            'skipped_serials' => $skipped,
            'total_received' => count($normalized),
        ]);
    }

    private function generateSku(string $name): string
    {
        $base = Str::upper(Str::slug($name, '-'));
        $rand = Str::upper(Str::random(6));
        return substr($base, 0, 8).'-'.$rand;
    }
}
