<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Exports\DailySalesExport;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class ReportController extends Controller
{
  public function profitAndLoss(Request $request)
  {
    $from = $request->query('from');
    $to   = $request->query('to');

    $fromDate = $from ? Carbon::parse($from)->startOfDay() : null;
    $toDate = $to ? Carbon::parse($to)->endOfDay() : null;

    $dateFilter = function ($q) use ($fromDate, $toDate) {
      if ($fromDate) $q->where('created_at', '>=', $fromDate);
      if ($toDate) $q->where('created_at', '<=', $toDate);
    };

    $income = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.type', 'income')
      ->where($dateFilter)
      ->selectRaw('SUM(ledger_entries.credit - ledger_entries.debit) as total')
      ->value('total') ?? 0;

    $expenses = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.type', 'expense')
      ->where($dateFilter)
      ->selectRaw('SUM(ledger_entries.debit - ledger_entries.credit) as total')
      ->value('total') ?? 0;

    return response()->json([
      'income' => round($income, 2),
      'expenses' => round($expenses, 2),
      'profit' => round($income - $expenses, 2),
    ]);
  }

  public function profitAndLossPdf(Request $request)
  {
    $data = $this->profitAndLoss($request)->getData(true);

    $pdf = Pdf::loadView('reports.pl', $data);

    return $pdf->download('profit-loss.pdf');
  }

  public function ledger(Request $request)
  {
    $accountCode = $request->query('account_code');
    $from = $request->query('from');
    $to   = $request->query('to');
    $fromDate = $from ? Carbon::parse($from)->startOfDay() : null;
    $toDate = $to ? Carbon::parse($to)->endOfDay() : null;

    $entries = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.code', $accountCode)
      ->when($fromDate, fn($q) => $q->where('ledger_entries.created_at', '>=', $fromDate))
      ->when($toDate, fn($q) => $q->where('ledger_entries.created_at', '<=', $toDate))
      ->orderBy('ledger_entries.created_at')
      ->get([
        'ledger_entries.created_at',
        'ledger_entries.debit',
        'ledger_entries.credit',
        'ledger_entries.description',
      ]);

    $balance = 0;
    $rows = [];

    foreach ($entries as $e) {
      $balance += ($e->debit - $e->credit);
      $rows[] = [
        'date' => $e->created_at,
        'debit' => $e->debit,
        'credit' => $e->credit,
        'balance' => round($balance, 2),
        'description' => $e->description,
      ];
    }

    return response()->json($rows);
  }

  public function vatReport(Request $request)
  {
    $from = $request->query('from');
    $to   = $request->query('to');
    $fromDate = $from ? Carbon::parse($from)->startOfDay() : null;
    $toDate = $to ? Carbon::parse($to)->endOfDay() : null;

    $vat = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.code', '2001')
      ->when($fromDate, fn($q) => $q->where('ledger_entries.created_at', '>=', $fromDate))
      ->when($toDate, fn($q) => $q->where('ledger_entries.created_at', '<=', $toDate))
      ->selectRaw('SUM(credit - debit) as vat_due')
      ->value('vat_due') ?? 0;

    return response()->json([
      'vat_due' => round($vat, 2),
    ]);
  }

  public function debtors()
  {
    $rows = DB::table('invoices')
      ->where('balance', '>', 0)
      ->select('invoice_no', 'customer_id', 'balance', 'created_at')
      ->orderBy('created_at', 'desc')
      ->get();

    return response()->json($rows);
  }

  public function dailySales(Request $request)
  {
    $from = $request->query('from');
    $to   = $request->query('to');
    $fromDate = $from ? Carbon::parse($from)->startOfDay() : null;
    $toDate = $to ? Carbon::parse($to)->endOfDay() : null;

    $sales = DB::table('invoices')
      ->whereIn('status', ['paid', 'partial'])
      ->when($fromDate, fn($q) => $q->where('created_at', '>=', $fromDate))
      ->when($toDate, fn($q) => $q->where('created_at', '<=', $toDate))
      ->selectRaw('DATE(created_at) as date, COUNT(*) as invoices, SUM(total) as total')
      ->groupByRaw('DATE(created_at)')
      ->orderBy('date')
      ->get();

    return response()->json($sales);
  }

  public function dailySalesExcel(Request $request)
  {
    return Excel::download(new DailySalesExport($request), 'daily-sales.xlsx');
  }

  public function closeMonth(Request $request)
  {
    DB::table('accounting_periods')->updateOrInsert(
      ['month' => $request->month],
      ['is_closed' => true]
    );

    return response()->json(['status' => 'closed']);
  }

  public function dashboard()
  {
    return response()->json([
      'today_sales' => DB::table('invoices')
        ->whereBetween('created_at', [now()->startOfDay(), now()->endOfDay()])
        ->sum('total'),
      'month_profit' => $this->profitForMonth(),
      'vat_due' => $this->vatDue(),
      'sales_trend' => $this->last7Days(),
    ]);
  }

  private function profitForMonth(): float
  {
    $from = now()->startOfMonth();
    $to = now()->endOfMonth();

    $income = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.type', 'income')
      ->whereBetween('ledger_entries.created_at', [$from, $to])
      ->selectRaw('SUM(ledger_entries.credit - ledger_entries.debit) as total')
      ->value('total') ?? 0;

    $expenses = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.type', 'expense')
      ->whereBetween('ledger_entries.created_at', [$from, $to])
      ->selectRaw('SUM(ledger_entries.debit - ledger_entries.credit) as total')
      ->value('total') ?? 0;

    return round($income - $expenses, 2);
  }

  private function vatDue(): float
  {
    $from = now()->startOfMonth();
    $to = now()->endOfMonth();

    $vat = DB::table('ledger_entries')
      ->join('accounts', 'accounts.id', '=', 'ledger_entries.account_id')
      ->where('accounts.code', '2001')
      ->whereBetween('ledger_entries.created_at', [$from, $to])
      ->selectRaw('SUM(credit - debit) as vat_due')
      ->value('vat_due') ?? 0;

    return round($vat, 2);
  }

  private function last7Days()
  {
    $from = now()->subDays(6)->startOfDay();
    $to = now()->endOfDay();

    return DB::table('invoices')
      ->whereBetween('created_at', [$from, $to])
      ->selectRaw('DATE(created_at) as date, COUNT(*) as invoices, SUM(total) as total')
      ->groupByRaw('DATE(created_at)')
      ->orderBy('date')
      ->get();
  }
}
